########### Filter data ###########
# This function takes cleaned data from Clean raw and will:
# - for most make this with and without unit cleaning!
# - Make line plot for hpopulation vector with permutation test
# - Make line plot split by Direction with permutation test
# - Heatmap of Direction across all units, seperate for Area.Brain/Treatment
# - Show bar plot for unit distribution (raw units plus percent), seperate for Area.Brain/Treatment


# Main function ---------------------------------------------------------------------------------------------------
plot_data <- function(config.file = './R/config.R'){
  tryCatch({
    
    # Load libraries
    suppressWarnings(library(tidyverse))
    suppressWarnings(library(futile.logger))
    suppressWarnings(library(pheatmap))
    
    suppressWarnings(library(furrr))
    plan(multiprocess) #multiprocess
    # Load config file
    source(config.file)
    source('./R/7_plot_data_functions.R')
    source('./R/additional_functions.R')
    
    # Set up logger
    flog.logger(name = log.name, threshold = threshold, appender = appender.tee(paste0(output.folder, log.name, ".log")))
    flog.threshold(threshold, name = log.name)
    
    flog.info(paste0("Initializing: ", format(Sys.Date(), "%d %B, %Y")), name = log.name)
    
    #### Load thresholded data ####
    flog.info("Loading thresholded data", name = log.name)
    Data <- read_rds(path = paste0(output.folder, Units.thresholded))
    
    if (!("UnitID" %in% colnames(Data))) {
      
      flog.info("Adding UnitID to thresholded data", name = log.name)
      Data <- Data %>% unite(UnitID, File.name, SessionID, Unit)
      
    }
    
    flog.info("Filtering data", name = log.name)
    Filtered.data <- Data %>%
      filter(Direction %in% c("Up", "Down", "Nonresponder"))
    
    flog.info("Loading one.sample permutation test", name = log.name)
    One.sample.observed <- read_rds(path = paste0(output.folder, one.sample.observed.file))
    One.sample.observed$time <- as.numeric(as.character(One.sample.observed$time))
    
    flog.info("Loading one.sample permutation test by direction", name = log.name)
    One.sample.observed.by.direction <- read_rds(path = paste0(output.folder, "By_direction_", 
                                                  one.sample.observed.file))
    One.sample.observed.by.direction$time <- as.numeric(as.character(One.sample.observed.by.direction$time))
    
    flog.info("Loading perievent data", name = log.name)
    Perievent <- read_rds(path = paste0(output.folder, Perievent.data))
    Perievent <- Perievent %>% 
      unite(UnitID, File.name, SessionID, Unit) %>% 
      select(-FileID, -ID)
    Perievent.mean <- calculate_mean(Perievent, permuted.stats.data = NULL, 
                                     group.cols = c("Area.brain", "TreatmentID", "Event", "UnitID", "norm.time"))
    Perievent.mean <- left_join(Perievent.mean, 
                                Filtered.data  %>% select(Area.brain, TreatmentID, Event, UnitID, Direction) %>%
                                  unique(),
                                by = c("Area.brain", "TreatmentID", "Event", "UnitID"))
    
    # Perievent.mean <- Perievent.mean %>%
    #   filter(Direction %in% c("Up", "Down", "Nonresponder"))
    
    #### Load time-wise data ####    
    flog.info("Loading time-wise data", name = log.name)
    Data.time <- read_rds(path = paste0(output.folder, cleaned.raw.data))
    
    
    if (!("UnitID" %in% colnames(Data.time))) {
      
      flog.info("Adding UnitID to time-wise data", name = log.name)
      Data.time <- Data.time %>% unite(UnitID, File.name, SessionID, Unit)
      
    }
    
    # select only cleaned units for precue poke firing
    flog.info("Filtering time-wise data", name = log.name)
    Data.time.filtered <- left_join(Data.time,
                                    Filtered.data %>% filter(Event == "Precue_Pokes") %>% 
                                      select(Area.brain, TreatmentID, FileID, UnitID, Direction) %>% unique(),
                                    by = c("Area.brain", "TreatmentID", "FileID", "UnitID"))
    Data.time.filtered <- Data.time.filtered %>% filter(!is.na(Direction))
    
    
    #### calculating mean ####
    flog.info("Calculatin mean time data", name = log.name)
    Data.mean.filtered <- calculate_mean(Filtered.data, One.sample.observed, 
                                         c("Area.brain", "TreatmentID", "Event", "norm.time"))

    flog.info("Calculatin mean time data by direction", name = log.name)
    Data.mean.filtered.by.Direction <- calculate_mean(Filtered.data, One.sample.observed.by.direction, 
                                                      c("Area.brain", "TreatmentID", "Event", "norm.time", "Direction")) 

    Data.time.mean <- Data.time.filtered %>%
      filter(Period == "Precue_period") %>% 
      group_by(UnitID, TrialCnt, Area.brain, TreatmentID) %>%
      summarise(z.score = mean(z.score),
                sum.pokes = sum(Precue_Pokes),
                n = n(), 
                precue.rate = sum.pokes/(n/4),
                Motion = mean(Motion),
                Direction = unique(Direction)) %>%
      ungroup()
              

    Data.time.mean$precue.rate_cut <- cut(Data.time.mean$precue.rate, 
                                         breaks = c(0, 0.01, 0.15, 0.3, 0.8),
                                         labels = c("None", "Low", "Mid", "High"),
                                         include.lowest = TRUE, right = FALSE)
              
    Data.time.mean.byUnit <- Data.time.mean %>% 
      group_by(UnitID, Area.brain, TreatmentID, precue.rate_cut) %>%
      summarise(z.score = mean(z.score),
                avg.pokes = mean(precue.rate),
                n = n(),
                Motion = mean(Motion),
                Direction = unique(Direction)) %>% 
      ungroup()
    
    #### Plot heatmaps ####
    
    # plot heatmap without cleaning (+/- time)
    flog.info("Plotting un-filtered data heatmap", name = log.name)
    plot <- NULL
    plot <- Data %>%
      select(-FileID) %>%
      group_by(Area.brain, TreatmentID) %>%
      nest() %>%
      unite(ID, Area.brain, TreatmentID) %>%
      mutate(ID = paste0("Non_filtered_", ID)) %>%
      mutate(plots = map(data, make_heatmap))
    
    walk2(plot$plots, plot$ID, save_plot, plots.folder = plots.folder, 20, 20)

    # plot tilemapwithout cleaning (+/- time)
    flog.info("Plotting un-filtered data tilemap", name = log.name)
    plot <- NULL
    plot <- Data %>%
      select(-FileID) %>%
      group_by(Area.brain, TreatmentID) %>%
      nest() %>%
      unite(ID, Area.brain, TreatmentID) %>%
      mutate(ID = paste0("Non_filtered_tilemap_", ID)) %>%
      mutate(plots = map(data, make_tilemap))
    
    walk2(plot$plots, plot$ID, save_plot, plots.folder = plots.folder, 20, 20)
    
    # plot heatmap with cleaning
    flog.info("Plotting filtered data heatmap", name = log.name)
    plot <- NULL
    plot <- Filtered.data %>% 
      group_by(Area.brain, TreatmentID) %>%
      nest() %>%
      unite(ID, Area.brain, TreatmentID) %>%
      mutate(ID = paste0("Filtered_", ID)) %>%
      mutate(plots = map(data, make_heatmap))
    
    walk2(plot$plots, plot$ID, save_plot, plots.folder = plots.folder, 20, 20)
    
    #### Plot single unit data ####
    
    # plot perievent single unit data
    # plot <- NULL
    # plot <- Perievent.mean %>% 
    #   group_by(Area.brain, TreatmentID, UnitID) %>%
    #   nest() %>%
    #   unite(ID, Area.brain, TreatmentID, UnitID) %>%
    #   mutate(ID = paste0("Raw_single_Unit_", ID)) %>%
    #   mutate(plots = map(data, single_unit_perievent))
    # 
    # future_map2(plot$plots, plot$ID, save_plot, plots.folder = plots.folder, 15, 15)
    
    # # plot unit Direction distribution without cleaning
    # flog.info("Plotting un-filtered single unit data", name = log.name)
    # plot <- NULL
    # plot <- Data %>% unite(UnitID, File.name, SessionID, Unit) %>% 
    #   select(-FileID) %>%
    #   group_by(Area.brain, TreatmentID, Event) %>%
    #   nest() %>%
    #   unite(ID, Area.brain, TreatmentID, Event) %>%
    #   mutate(ID = paste0("Non_filtered_", ID)) %>%
    #   mutate(plots = map(data, single_unit_plot))
    # 
    # future_map2(plot$plots, plot$ID, save_plot, plots.folder = plots.folder, 15, 15)

    # # plot unit Direcion distribution with cleaning
    # flog.info("Plotting filtered single unit data", name = log.name)
    # plot <- NULL
    # plot <- Filtered.data %>% 
    #   group_by(Area.brain, TreatmentID, Event) %>%
    #   nest() %>%
    #   unite(ID, Area.brain, TreatmentID, Event) %>%
    #   mutate(ID = paste0("Filtered_single_unit", ID)) %>%
    #   mutate(plots = map(data, single_unit_plot))
    # 
    # walk2(plot$plots, plot$ID, save_plot, plots.folder = plots.folder, 15, 15)
    
    #### Plotting mean data ####
    
    # plot time plots for population with one sample t.test
    flog.info("Plotting population data", name = log.name)
    plot <- NULL
    plot <- Data.mean.filtered %>% 
      group_by(Area.brain, TreatmentID) %>%
      nest() %>%
      unite(ID, Area.brain, TreatmentID) %>%
      mutate(ID = paste0("Population_vector_", ID)) %>%
      mutate(plots = map(data, population_plot, by.direction = FALSE))
    
    walk2(plot$plots, plot$ID, save_plot, plots.folder = plots.folder, 15, 10)
    
    # plot time plots for Direction with one sample t.test
    flog.info("Plotting population data by direction", name = log.name)
    plot <- NULL
    plot <- Data.mean.filtered.by.Direction %>% 
      group_by(Area.brain, TreatmentID) %>%
      nest() %>%
      unite(ID, Area.brain, TreatmentID) %>%
      mutate(ID = paste0("Population_vector_byDirection_", ID)) %>%
      mutate(plots = map(data, population_plot, by.direction = TRUE))
    
    walk2(plot$plots, plot$ID, save_plot, plots.folder = plots.folder, 20, 6)

    flog.info("It is done!", name = log.name)
    
    #### Unit distribution ####
    flog.info("Plotting raw distrubution of units", name = log.name)
    plot <- Data %>% 
      calculate_distribution() %>%
      distribution_plot()
    
    save_plot(Plot = plot, ID = "Non_filtered_Distribution_plot.png", plots.folder = plots.folder, 15, 10)
    
    flog.info("Plotting filtered distrubution of units", name = log.name)
    plot <- Filtered.data %>% 
      calculate_distribution() %>%
      distribution_plot()
    
    save_plot(Plot = plot, ID = "Filtered_Distribution_plot.png", plots.folder = plots.folder, 15, 10)
    
    #### Firing in precue poke #### 
    flog.info("Plotting firing depending on precue pokes", name = log.name)
    plot <- Data.time.mean.byUnit %>% 
      group_by(Area.brain, TreatmentID) %>%
      nest() %>%
      unite(ID, Area.brain, TreatmentID) %>%
      mutate(ID = paste0("Precue_pokes_VS_firing_byDirection_ByUnit_", ID)) %>%
      mutate(plots = map(data, scatter_plot))
    
    walk2(plot$plots, plot$ID, save_plot, plots.folder = plots.folder, 20, 15)
    
    plot <- Data.time.mean %>% 
      group_by(Area.brain, TreatmentID) %>%
      nest() %>%
      unite(ID, Area.brain, TreatmentID) %>%
      mutate(ID = paste0("Precue_pokes_VS_firing_byDirection_ByTrial_", ID)) %>%
      mutate(plots = map(data, scatter_plot, x_axis = "precue.rate"))
    
    walk2(plot$plots, plot$ID, save_plot, plots.folder = plots.folder, 20, 15) 
    
    
  }, error = function(e) {
    flog.error("ERROR!", name = log.name)
    flog.error(e, name = log.name)
  })
}
